using System;
using System.Drawing;

namespace SimpleXB.Drawing
{
	public struct CanvasElement
	{
		byte character;
		byte attribute;

		public CanvasElement(byte character, byte attribute)
		{
			this.character = character;
			this.attribute = attribute;
		}

		public byte Character
		{
			get { return character; }
			set { character = value; }
		}

		public byte Attribute
		{
			get { return attribute; }
			set { attribute = value; }
		}

		public ushort Value
		{
			get { return (ushort)(character + attribute << 8); }
			set 
			{
				character = (byte)(value & 0xFF);
				attribute = (byte)(value >> 8);
			}
		}

		public byte Foreground
		{
			get { return (byte)(attribute & 0x0F); }
			set { attribute = (byte)((attribute & 0xF0) | (value & 0x0f)); }
		}

		public byte Background
		{
			get { return (byte)(attribute >> 4); }
			set { attribute = (byte)((attribute & 0x0F) | ((value & 0x0F) << 4)); }
		}

	}
	
	/// <summary>
	/// Summary description for Canvas.
	/// </summary>
	public abstract class Canvas
	{
		private Size canvasSize;
		public Canvas(Size canvasSize)
		{
			this.canvasSize = canvasSize;
		}

		public abstract CanvasElement this[int x, int y] 
		{
			get; set;
		}

		public virtual void Clear()
		{
			Clear(new CanvasElement(32, 7));
		}

		public abstract void Clear(CanvasElement ce);

		public virtual Size CanvasSize
		{
			get { return canvasSize; }
			set { ResizeCanvas(value); canvasSize = value; }
		}

		protected abstract void ResizeCanvas(Size newCanvasSize);
	}


	public class MemoryCanvas : Canvas
	{
		CanvasElement ceDefault;
		private CanvasElement[][] canvas;
		public MemoryCanvas(Size canvasSize) : base(canvasSize)
		{
			ceDefault = new CanvasElement(32,7);
			canvas = new CanvasElement[canvasSize.Height][];
			for (int i=0; i<canvasSize.Height; i++)
			{
				canvas[i] = null;
			}
		}

		public override void Clear(CanvasElement ce)
		{
			ceDefault = ce;
			for (int i=0; i<CanvasSize.Height; i++)
			{
				canvas[i] = null;
			}
		}

		public override CanvasElement this[int x, int y]
		{
			get 
			{
				CanvasElement[] line = canvas[y];
				return (line == null) ? new CanvasElement(32, 7) : line[x];
			}
			set 
			{
				CanvasElement[] line = canvas[y];
				if (line == null) canvas[y] = line = new CanvasElement[CanvasSize.Width];
				line[x] = value;
			}
		}

		protected override void ResizeCanvas(Size newCanvasSize)
		{
			canvas = new CanvasElement[newCanvasSize.Height][];
			for (int i=0; i<newCanvasSize.Height; i++)
			{
				canvas[i] = null;
			}
			// need to copy over old info later.. maybe using a parameter to specify?
		}
	}
}
