using System;
using System.IO;
using System.Drawing;
using System.Text;
using SimpleXB.Drawing;
using System.Runtime.InteropServices;

namespace SimpleXB.Formats
{
	/// <summary>
	/// Summary description for Format.
	/// </summary>
	public abstract class Format
	{
		public abstract void Load(Stream fs, Page page);
	}

	public class FormatXBin : Format
	{
		private const string XBIN_ID = "XBIN";

		private struct XBinHeader
		{
			public byte[] xbinID;
			public byte eofChar;
			public ushort width;
			public ushort height;
			public byte fontSize;
			public byte flags;

			public XBinHeader(BinaryReader br)
			{
				xbinID = br.ReadBytes(4);
				eofChar = br.ReadByte();
				width = br.ReadUInt16();
				height = br.ReadUInt16();
				fontSize = br.ReadByte();
				flags = br.ReadByte();
			}

			public string XBinID
			{
				get  { return ASCIIEncoding.ASCII.GetString(xbinID); }
			}

			public bool Palette { get { return (flags & 0x01) != 0; } }
			public bool Font { get { return (flags & 0x02) != 0; } }
			public bool Compress { get { return (flags & 0x04) != 0; } }
			public bool NonBlink { get { return (flags & 0x08) != 0; } }
			public bool Font512 { get { return (flags & 0x10) != 0; } }
		}

		public override void Load(Stream fs, Page page)
		{
			BinaryReader br = new BinaryReader(fs);
			XBinHeader xbh = new XBinHeader(br);
			if (xbh.XBinID != XBIN_ID) throw new FileLoadException("not a valid XBIN file");
			Size loadSize = new Size(0,0);
			Canvas c = page.Canvas;
			c.CanvasSize = new Size(xbh.width, xbh.height);

			loadSize.Width = (xbh.width > c.CanvasSize.Width) ? c.CanvasSize.Width : xbh.width;
			loadSize.Height = (xbh.height > c.CanvasSize.Height) ? c.CanvasSize.Height : xbh.height;

			page.Palette = Palette.GetDefaultPalette();
			if (xbh.Palette)
			{
				page.Palette.Load(br, 16);
			}

			if (xbh.Font)
			{
				SimpleXB.Drawing.Font f = new SimpleXB.Drawing.Font(xbh.Font512 ? 512 : 256, 8, xbh.fontSize);
				f.Load(br);
				page.Font = f;
			}

			page.NonBlink = xbh.NonBlink;

			CanvasElement ce = new CanvasElement(32, 7);
			if (xbh.Compress)
			{
				for (int y=0; y<loadSize.Height; y++)
				{
					int x = 0;
					while (x<xbh.width)
					{
						int countbyte = br.ReadByte();
						if (countbyte == -1) break;
						int runlength = (countbyte & 0x3F) + 1;
						switch (countbyte & 0xc0)
						{
							case 0x00:
								while (runlength>0)
								{
									ce.Character = br.ReadByte();
									ce.Attribute = br.ReadByte();
									if (x<c.CanvasSize.Width) c[x,y] = ce;
									x++;
									runlength--;
								}
								break;
							case 0x40:
								ce.Character = br.ReadByte();
								while (runlength>0)
								{
									ce.Attribute = br.ReadByte();
									if (x < c.CanvasSize.Width) c[x,y] = ce;
									x++;
									runlength--;
								}
								break;
							case 0x80:
								ce.Attribute = br.ReadByte();
								while (runlength>0) 
								{
									ce.Character = br.ReadByte();
									if (x < c.CanvasSize.Width) c[x,y] = ce;
									x++;
									runlength--;
								}
								break;
							case 0xc0:
								ce.Character = br.ReadByte();
								ce.Attribute = br.ReadByte();
								while (runlength>0)
								{
									if (x < c.CanvasSize.Width) c[x,y] = ce;
									x++;
									runlength--;
								}
								break;
						}
					}
				}
			}
			else
			{
				for (int y=0; y<loadSize.Height; y++)
				{
					for (int x=0; x<xbh.width; x++)
					{
						ce.Character = br.ReadByte();
						ce.Attribute = br.ReadByte();
						if (x < c.CanvasSize.Width) c[x,y] = ce;
					}
				}
			}
		}
	}
}

