using System;
using System.IO;
using System.Text;

namespace SimpleXB.Sauce
{
	public class SauceFlag
	{
		private string description;
		private Sauce sauce;
		private int flag;

		public SauceFlag(Sauce sauce, int flag, string description)
		{
			this.description = description;
			this.flag = flag;
			this.sauce = sauce;
		}

		public string Description
		{
			get { return description; }
			set { description = value; }
		}
		private byte Mask
		{
			get { return (byte)(0x01 << flag); }
		}

		public bool Value
		{
			get { return (sauce.ByteFlags & Mask) != 0; }
			set 
			{
				if (value) sauce.ByteFlags |= Mask;
				else sauce.ByteFlags &= (byte)~Mask;
			}
		}
	}

	public enum SauceDataType
	{
		None = 0,		// Undefined filetype, you could use this to add SAUCE information to personal datafiles needed by programs, but not having any other meaning. 
		Character = 1,	// Any character based file. Examples are ASCII, ANSi and RIP. 
		Graphics = 2,	// Any bitmap graphic file. Examples are GIF, LBM, and PCX. 
		Vector = 3,		// Any vector based graphic file. Examples are DXF and CAD files.
		Sound = 4,		// Any sound related file. Examples are samples, MOD files and MIDI. 
		BinaryText = 5,	// This is RAW memory copy of a text screen. It's basically the BIN format you can save from whitin TheDraw. Each character is built up of two consecutive bytes. The first is the character to be displayed. The second is the Attribute byte. 
		XBIN = 6,		// XBIN is the so called eXtended BIN format. It is similar to the BinaryText, but provides for fonts, palettes, and has built-in compression. 
		Archive = 7,	// Any type of archive. Examples are ARC, ZIP, ARJ and LZH. 
		Executable = 8	// Any file that is executable. 
	}

	/// <summary>
	/// Summary description for Sauce.
	/// </summary>
	public class Sauce
	{
		private const string SAUCE_ID = "SAUCE";
		private const long SAUCE_SIZE = 128;

		private byte[] id;
		private byte[] version;
		private byte[] title;
		private byte[] author;
		private byte[] group;
		private byte[] date;
		private Int32 fileSize;
		private byte dataType;
		private byte fileType;
		private UInt16 tinfo1;
		private UInt16 tinfo2;
		private UInt16 tinfo3;
		private UInt16 tinfo4;
		private byte numComments;
		private byte flags;
		private SauceComment comments;
		private SauceDataTypeInfo typeInfo;

		public static bool SauceExists(Stream s)
		{
			BinaryReader br = new BinaryReader(s);
			long pos = s.Position;
			s.Seek(s.Length - SAUCE_SIZE, SeekOrigin.Begin);
			byte[] id = br.ReadBytes(5);

			bool exists = (ASCIIEncoding.ASCII.GetString(id) == SAUCE_ID);
			
			// restore original position
			s.Seek(pos, SeekOrigin.Begin);
			return exists;
		}
		
		public Sauce(Stream stream, bool seekPosition)
		{
			long origin = stream.Position;
			try 
			{
				BinaryReader br = new BinaryReader(stream);
				if (seekPosition) br.BaseStream.Seek(stream.Length - SAUCE_SIZE, SeekOrigin.Begin);
				long start = br.BaseStream.Position;
				id = br.ReadBytes(5);
				if (ID != SAUCE_ID) throw new Exception("Sauce does not exist!");
				version = br.ReadBytes(2);
				title = br.ReadBytes(35);
				author = br.ReadBytes(20);
				group = br.ReadBytes(20);
				date = br.ReadBytes(8);
				fileSize = br.ReadInt32();
				dataType = br.ReadByte();
				SetTypeInfo();
				fileType = br.ReadByte();
				tinfo1 = br.ReadUInt16();
				tinfo2 = br.ReadUInt16();
				tinfo3 = br.ReadUInt16();
				tinfo4 = br.ReadUInt16();
				numComments = br.ReadByte();
				flags = br.ReadByte();

				br.BaseStream.Seek(start-(numComments*SauceComment.COMMENT_SIZE)-5, SeekOrigin.Begin);
				comments = new SauceComment(br, numComments);
			}
			finally
			{
				stream.Seek(origin, SeekOrigin.Begin);
			}
		}

		private void SetTypeInfo()
		{
			switch (DataType)
			{
				default:
				case SauceDataType.None: 
					typeInfo = new SauceDataTypeInfo(this);
					break;
				case SauceDataType.Character:
					typeInfo = new CharacterDataTypeInfo(this);
					break;
				case SauceDataType.XBIN:
					typeInfo = new XBinDataTypeInfo(this);
					break;
			}
		}


		public string ID
		{
			get { return ASCIIEncoding.ASCII.GetString(id); }
			set { id = ASCIIEncoding.ASCII.GetBytes(value); }
		}

		public int Version
		{
			get 
			{
				string s = ASCIIEncoding.ASCII.GetString(version);
				return Convert.ToInt32(s);
			}
			set 
			{
				string s = value.ToString().PadLeft(2, '0');
				if (s.Length > 2) throw new Exception("version must not exceed 2 digits");
				version = ASCIIEncoding.ASCII.GetBytes(s);
			}
		}

		public string Title
		{
			get { return ASCIIEncoding.ASCII.GetString(title).TrimEnd(); }
			set 
			{
				string s = value.PadRight(35);
				if (s.Length > 35) throw new Exception("title too long");
				title = new byte[35];
				ASCIIEncoding.ASCII.GetBytes(s, 0, 35, title, 0);
			}
		}

		public string Author
		{
			get { return ASCIIEncoding.ASCII.GetString(author).TrimEnd(); }
			set 
			{
				string s = value.PadRight(20);
				if (s.Length > 20) throw new Exception("author too long");
				author = ASCIIEncoding.ASCII.GetBytes(s);
			}
		}

		public string Group
		{
			get { return ASCIIEncoding.ASCII.GetString(group).TrimEnd(); }
			set 
			{
				string s = value.PadRight(20);
				if (s.Length > 20) throw new Exception("group too long");
				group = ASCIIEncoding.ASCII.GetBytes(s);
			}
		}

		public DateTime Date
		{
			get 
			{
				string s = ASCIIEncoding.ASCII.GetString(date);
				int year = Convert.ToInt32(s.Substring(0, 4));
				int month = Convert.ToInt32(s.Substring(4, 2));
				int day = Convert.ToInt32(s.Substring(6, 2));

				return new DateTime(year, month, day);
			}
			set 
			{
				string s = string.Format("{0,4}{1,2}{2,2}", value.Year, value.Month, value.Day);
				date = ASCIIEncoding.ASCII.GetBytes(s);
			}
		}

		public int FileSize
		{
			get { return fileSize; }
			set { fileSize = value; }
		}

		internal byte ByteFileType
		{
			get { return fileType; }
			set { fileType = value; }
		}

		public byte ByteFlags
		{
			get { return flags; }
			set { flags = value; }
		}

		public SauceFlag[] Flags
		{
			get { return typeInfo.Flags; }
		}

		public string[] Comments
		{
			get { return comments.Comments; }
			//set { comments.Comments = value; }
		}

		public SauceDataType DataType
		{
			get { return (SauceDataType)dataType; }
			set 
			{
				dataType = (byte)value;
				SetTypeInfo();
			}
		}

		public string FileTypeDescription
		{
			get { return typeInfo.FileTypeDescription; }
		}
		public virtual string TInfo1Description { get { return typeInfo.GetTInfoDescription(1); } }
		public virtual string TInfo2Description { get { return typeInfo.GetTInfoDescription(2); } }
		public virtual string TInfo3Description { get { return typeInfo.GetTInfoDescription(3); } }
		public virtual string TInfo4Description { get { return typeInfo.GetTInfoDescription(4); } }

		public UInt16 TInfo1
		{
			get { return this.tinfo1; }
			set { this.tinfo1 = value; }
		}
		public UInt16 TInfo2
		{
			get { return this.tinfo2; }
			set { this.tinfo2 = value; }
		}
		public UInt16 TInfo3
		{
			get { return this.tinfo3; }
			set { this.tinfo3 = value; }
		}
		public UInt16 TInfo4
		{
			get { return this.tinfo4; }
			set { this.tinfo4 = value; }
		}
	
	}

	public class SauceComment
	{
		public const string COMMENT_ID = "COMNT";
		public const int COMMENT_SIZE = 64;
		private byte[] id;
		private byte[][] data;

		public SauceComment(BinaryReader br, int numComments)
		{
			id = br.ReadBytes(5);
			if (ID == COMMENT_ID)
			{
				data = new byte[numComments][];
				for (int i=0; i<numComments; i++)
				{
					data[i] = br.ReadBytes(COMMENT_SIZE);
				}
			}
			else data = new byte[0][];
		}

		private string ID
		{
			get { return ASCIIEncoding.ASCII.GetString(id); }
			set { id = ASCIIEncoding.ASCII.GetBytes(value); }
		}

		public string[] Comments
		{
			get 
			{
				string[] comments = new string[data.Length];
				for (int i=0; i<data.Length; i++)
				{
					comments[i] = ASCIIEncoding.ASCII.GetString(data[i]);
				}
				return comments;
			}
		}
	}
}
